const express = require("express");
const multer = require("multer");
const path = require("path");

const db = require("../models");

const BoardUser = db.board_user;

const upload_storage = multer.diskStorage({
  destination: function (req, file, cb) {
    cb(null, "uploads/board_cv");
  },
  filename: function (req, file, cb) {
    cb(null, Date.now() + path.extname(file.originalname));
  },
});

const uploadStorage = multer({ storage: upload_storage }).fields([
  { name: "user_pdf", maxCount: 1 },
  { name: "user_pdf_en", maxCount: 1 },
]);

const getBoardUsers = async (req, res) => {
  try {
    const board_users = await BoardUser.findAll();

    res.status(200).send(board_users);
  } catch (err) {
    res.status(404).send(err.message);
  }
};

const createBoardUser = async (req, res) => {
  const {
    full_name,
    department,
    isTopLevel,
    isBordiShkencor,
    isBordiEditorial,
    isRedaksia,
    isKomitetiOrganizativ,
    isAnetar,
  } = req.body;

  const { user_pdf, user_pdf_en } = req.files;

  // Validate required fields
  if (!full_name && !user_pdf && !user_pdf_en) {
    return res
      .status(400)
      .json({ error: "Please provide the Full Name and CV file!" });
  }

  try {
    const full_data = {
      full_name,
      department,
      cv_pdf: path.join("uploads/board_cv", user_pdf[0].filename),
      cv_en_pdf: path.join("uploads/board_cv", user_pdf_en[0].filename),
      isTopLevel,
      isBordiShkencor,
      isBordiEditorial,
      isRedaksia,
      isKomitetiOrganizativ,
      isAnetar,
    };
    const board_user = await BoardUser.create(full_data);

    return res.status(201).json({
      message: "Board User has been created successfully.",
      board_user,
    });
  } catch (err) {
    res.status(500).send(err.message);
  }
};

const updateBoardUser = async (req, res) => {
  const { id } = req.params;
  const {
    full_name,
    department,
    isTopLevel,
    isBordiShkencor,
    isBordiEditorial,
    isRedaksia,
    isKomitetiOrganizativ,
    isAnetar,
  } = req.body;

  const { user_pdf, user_pdf_en } = req.files;

  let full_data = {
    full_name,
    department,
    isTopLevel,
    isBordiShkencor,
    isBordiEditorial,
    isRedaksia,
    isKomitetiOrganizativ,
    isAnetar,
  };

  if (user_pdf && user_pdf_en) {
    full_data.cv_pdf = path.join("uploads/board_cv", user_pdf[0].filename);
    full_data.cv_en_pdf = path.join("uploads/board_cv", user_pdf_en[0].filename);
  }

  try {
    const current_user = await BoardUser.findByPk(id);
    const updated_user = await current_user.update(full_data);
    res.status(200).json({
      message: "Board User has been updated successfully!",
      material: updated_user,
    });
  } catch (err) {
    return res.status(400).send({ error: err.message });
  }
};

const deleteBoardUser = async (req, res) => {
  try {
    const result = await BoardUser.destroy({
      where: { id: req.params.id },
    });

    if (result == 1) {
      res.status(200).send("Board User has been successfully deleted!");
    } else {
      res.status(404).send("Board User doesn't exist!");
    }
  } catch (err) {
    return res.status(err.status).send({ error: err.message });
  }
};

module.exports = {
  getBoardUsers,
  createBoardUser,
  deleteBoardUser,
  updateBoardUser,
  uploadStorage,
};
