const express = require('express')
const multer = require('multer');
const path = require("path")


const db = require('../models');

const Materials = db.materials;
const Categories = db.categories;

const upload_storage = multer.diskStorage({
    destination: function (req, file, cb) {
        cb(null, 'uploads/materials');
    },
    filename: function (req, file, cb) {
        cb(null, Date.now() + path.extname(file.originalname))
    }
})

const uploadFields = multer({ storage: upload_storage }).fields([
    { name: 'image', maxCount: 1 },
    { name: 'pdf', maxCount: 1 }
])

const getMaterials = async (req, res) => {
    try {
        const _materials = await Materials.findAll();

        res.status(200).send(_materials);
    } catch (err) {
        res.status(404).send(err.message)
    }
}

const createMaterial = async (req, res) => {
    const { title, short_description, publish_number, publish_year, categoryId, author } = req.body;
    const { image, pdf } = req.files;


    // Validate required fields
    if (!title || !image || !pdf) {
        return res.status(400).json({ error: 'Please provide a Title, Image and a PDF file!' });
    }

    try {

        if (!categoryId) {
            return res.status(400).json({ error: "Category not found." })
        }

        const material = {
            title,
            short_description,
            publish_number,
            publish_year,
            image: image[0].filename,
            pdf: pdf[0].filename,
            categoryId: categoryId,
            authors: author
        };
        const _material = await Materials.create(material)


        return res.status(201).json({ message: 'Material has been created successfully.', _material });

    } catch (err) {
        res.status(500).send(err.message)
    }
}

const updateMaterial = async (req, res) => {
    const { id } = req.params;
    const { title, short_description, publish_number, publish_year, categoryId, author } = req.body;
    const { image, pdf } = req.files;

    let new_data = {
        title: title,
        short_description: short_description,
        publish_number: publish_number,
        publish_year: publish_year,
        categoryId: categoryId,
        authors: author
    }

    if (image) {
        new_data = {
            title: title,
            short_description: short_description,
            publish_number: publish_number,
            publish_year: publish_year,
            image: image[0]?.filename,
            categoryId: categoryId,
            authors: author
        }
    }

    if (pdf) {
        new_data = {
            title: title,
            short_description: short_description,
            publish_number: publish_number,
            publish_year: publish_year,
            pdf: pdf[0]?.filename,
            categoryId: categoryId,
            authors: author
        }
    }



    try {
        const current_material = await Materials.findByPk(id);
        const updated_material = await current_material.update(new_data);
        res.status(200).json({ message: "Material updated successfully!", material: updated_material })
    } catch (err) {
        return res.status(400).send({ error: err.message })
    }
}

const deleteMaterial = async (req, res) => {
    try {
        const result = await Materials.destroy({
            where: { id: req.params.id }
        })

        if (result == 1) {
            res.status(200).send("Material has been successfully deleted!");
        } else {
            res.status(404).send("Material doesn't exist!")
        }

    } catch (err) {
        return res.status(err.status).send({ error: err.message })
    }
}

module.exports = {
    getMaterials,
    createMaterial,
    deleteMaterial,
    updateMaterial,
    uploadFields
}