const express = require("express");
const multer = require("multer");
const path = require("path");

const db = require("../models");

const Call = db.calls;
const CallFile = db.callFiles;

const files_storage = multer.diskStorage({
  destination: function (req, file, cb) {
    cb(null, "uploads/call-materials");
  },
  filename: function (req, file, cb) {
    cb(null, Date.now() + path.extname(file.originalname));
  },
});

const uploadFileStorage = multer({
  storage: files_storage,
  limits: { files: 6 },
}).array("word_files", 6);

const getCalls = async (req, res) => {
  try {
    const calls = await Call.findAll({
      include: [
        {
          model: CallFile,
          as: "call_files",
        },
      ],
    });

    res.status(200).send(calls);
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
};

const createCall = async (req, res) => {
  const { title, short_description, description, en_description, end_date } =
    req.body;

  if (
    !title ||
    !short_description ||
    !description ||
    !en_description ||
    !end_date
  ) {
    return res.status(400).json({
      error:
        "Please provide Title , Short Description, Full Description & End Date",
    });
  }

  try {
    let call_data = {
      title: title,
      short_description: short_description,
      en_description: en_description,
      description: description,
      end_date: end_date,
    };

    const call = await Call.create(call_data);

    const word_files = req.files.map((file) => ({
      url: path.join("uploads/call-materials", file.filename),
      title: file.originalname,
      call_id: call.id,
    }));

    await CallFile.bulkCreate(word_files);

    res.status(201).send(call);
  } catch (err) {
    res.status(500).send(err.message);
  }
};

const deleteCall = async (req, res) => {
  try {
    const result = await Call.destroy({
      where: { id: req.params.id },
    });

    if (result == 1) {
      res.status(200).send("Call has been successfully deleted!");
    } else {
      res.status(404).send("Call doesn't exist!");
    }
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
};

const updateCall = async (req, res) => {
  const { id } = req.params;
  const { title, short_description, description, en_description, end_date } =
    req.body;

  let updated_data = {
    title: title,
    short_description: short_description,
    description: description,
    en_description: en_description,
    end_date: end_date,
  };

  const word_files = req.files;

  try {
    const current_call = await Call.findByPk(id);
    const updatedCall = await current_call.update(updated_data);

    if (word_files && word_files.length > 0) {
      await CallFile.destroy({ where: { call_id: id } });

      const new_files = req.files.map((file) => ({
        url: path.join("uploads/call-materials", file.filename),
        title: file.originalname,
        call_id: id,
      }));

      await CallFile.bulkCreate(new_files);
    }

    res
      .status(200)
      .json({ message: "Call has been updated successfully!", updatedCall });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
};

module.exports = {
  createCall,
  updateCall,
  deleteCall,
  getCalls,
  uploadFileStorage,
};
